/************************************************************************
 * @file: AudioSilenceDetector.cpp
 *
 * @description: This cpp file contains definition for class SilenceDetector.
 * SilenceDetector is a Post Processing utility to detect the silence introduced
 * in PCM data. Silence detector will warn only if silence occurred between continous
 * valid pcm data. Initial Silence till valid data comes and silence data after proper
 * fade out will be neglected.
 * Silence detector process does sampling of pcm data in to small chunks [2ms] and
 * mean is calculated for the second of the chunk and based on past amd current mean values
 * fade and silence will be detected.
 *
 * @authors: Jens Lorenz, jlorenz@de.adit-jv.com 2016
 *           Vijay Palaniswamy, vijay.palaniswamy@in.bosch.com 2016
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/
#include <algorithm>
#include <cstring>
#include <endian.h>
#include <cstdlib>


#include "AudioSilenceDetector.h"
#include "AudioHelper.h"

using namespace adit::utility::audio;
using namespace std;

#if __BYTE_ORDER == __LITTLE_ENDIAN
#define SD_LITTLE_ENDIAN
#elif __BYTE_ORDER == __BIG_ENDIAN
#define SD_BIG_ENDIAN
#else
#error "Unsupported endian..."
#endif

#define SD_FALSE 0
#define SD_TRUE  1

SilenceDetector::SilenceDetector(Streaming& streamingHandle) :  mStreamingHandle(streamingHandle)
{
    defaults();
}

SilenceDetector::~SilenceDetector()
{
    clearSdPrivateData();
}

SdRet SilenceDetector::configure(const SdStreamInfo streamInfo)
{
    SdRet ret;

    if (getState() != SdState::NONE)
    {
        return SdRet::FAILURE;
    }

    mFormat         = streamInfo.format;
    mChannels       = streamInfo.channels;
    mRate           = streamInfo.rate;
    mFrames         = streamInfo.frames;
    mBytesPerSample = getBytesPerSample(mFormat);
    mFrameSize      = mChannels * mFrames * mBytesPerSample;

    ret = allocSilenceBuffer();
    if (ret != SdRet::OK)
    {
        clearSdPrivateData();
        return  ret;
    }

    ret = isFormatSupported(streamInfo.format);
    if (ret != SdRet::OK)
    {
        clearSdPrivateData();
        return ret;
    }



    ret = updateSdPrivateData();
    if (ret == SdRet::OK)
    {
        changeState(SdState::CONFIGURED);
    }
    else
    {
        clearSdPrivateData();
    }

    return ret;
}

SdRet SilenceDetector::process(const void* buf, const uint32_t frames)
{
    int32_t rc = 0;
    uint32_t remain = frames;

    if (getState() == SdState::NONE)
    {
        return SdRet::FAILURE;
    }

    if (buf == nullptr)
    {
        stopProcessing();
        return SdRet::FAILURE;
    }

    if (frames < mMinFramesReq)
    {
        stopProcessing();
        return SdRet::FAILURE;
    }

    while (remain != 0)
    {
        uint32_t processFrames = 0;
        if (remain > mFrames)
        {
            processFrames = mFrames;
        }
        else
        {
            processFrames = remain;
        }

        rc = memcmp(buf, static_cast<void *>(mSilenceBuf), (processFrames * mChannels * getBytesPerSample(mFormat)));
        if (rc != 0)
        {
            break;
        }

        remain = remain - processFrames;
    }

    if (mFirstActiveSampleDone)
    {
        if (rc == 0)
        {
            /* Full period is silence */
            isSilence();
        }
        else
        {
            processInternal(buf, frames);
        }
    }
    else
    {
        /* Start check for silence only after valid data*/
        if (rc != 0)
        {
            processInternal(buf, frames);
        }
    }

    /* Remove mean values exceeding the limit*/
    while (mMean.size() > mNoOfMeanValues)
    {
        mMean.pop_back();
    }

    return SdRet::OK;
}

void SilenceDetector::activate()
{
    mFirstActiveSampleDone = 0;
    mMean.clear();
    changeState(SdState::STOPPED);
}

void SilenceDetector::reset()
{
    clearSdPrivateData();
}

SdRet SilenceDetector::processInternal(const void* buf, const uint32_t frames)
{
    SdRet ret = SdRet::OK;
    switch (mFormat)
    {
        case S8:
	        ret = processData(static_cast<const int8_t*>(buf), frames);
            break;
        case U8:
            ret = processData(static_cast<const uint8_t*>(buf), frames);
            break;
        case S16_LE:
            ret = processData(static_cast<const int16_t*>(buf), frames);
            break;
        case S16_BE:
            ret = processData(static_cast<const int16_t*>(buf), frames);
            break;
        case U16_LE:
            ret = processData(static_cast<const uint16_t*>(buf), frames);
            break;
        case U16_BE:
            ret = processData(static_cast<const uint16_t*>(buf), frames);
            break;
        default :
            return SdRet::UNSUPPORTED;
    }
    return ret;
}

template<typename T>
SdRet SilenceDetector::processData(const T* buf, const uint32_t frames)
{
    uint32_t loopCount = 0;

    /* Split total frames in to chunks of 2 mili seconds */
    loopCount = frames / mMinFramesReq;
    for (uint32_t i = 0 ; i < loopCount; i++)
    {
        uint32_t mean = 0;
        /*Analyzing the second half of each chunk*/
        const T *temp = buf + (((i * mMinFramesReq) + (mMinFramesReq >> 1)) * mChannels);
        mean = calculateMean(temp, (mMinFramesReq >> 1));

        /* During fading some times mean value is zero or near to zero */
        if ((getState() == SdState::FADING) && (mean <= mMinMeanAmplitude))
        {
            mean = 0;
        }

        if (mean == 0)
        {
            isSilence();
        }
        else
        {
            if (!mFirstActiveSampleDone)
            {
                startProcessing();
            }

            addtoMeanlist(mean);
        }
    }
    return SdRet::OK;
}

SdRet SilenceDetector::checkForFading()
{
    SdRet ret = SdRet::OK;

    if (mMean.size() < mNoOfMeanValues)
    {
        return SdRet::FAILURE;
    }

    int32_t srtRet = is_sorted(mMean.begin(),mMean.begin() + mNoOfMeanValues);

    if (!srtRet)
    {
        ret = SdRet::FAILURE;
    }

    return ret;
}

void SilenceDetector::isSilence()
{
    if (getState() == SdState::STOPPED)
    {
        return;
    }

    if (checkForFading() == SdRet::OK)
    {
        changeState(SdState::FADING);
        mFirstActiveSampleDone = 0;
    }
    else
    {
        Logging(mStreamingHandle, LL_WARNING) << "SilenceDetector::isSilence Silence detected" << Logging::endl;
        stopProcessing();
    }
}


void SilenceDetector::addtoMeanlist(const uint32_t mean)
{
    uint32_t ret = mean;
    if (mMean.size() > 0)
    {
        if (mean > mMean.front())
        {
            uint32_t diff = mean - mMean.front();
            if (diff <= DIFF_TOLERANCE)
            {
                ret = mMean.front();
            }
        }
    }
    mMean.insert(mMean.begin(), ret);
}

template<typename M>
uint32_t SilenceDetector::calculateMean(const M* buf, const uint32_t frames)
{
    uint64_t total = 0;
    uint32_t mean = 0;
    M value = 0;
    bool swap = isSwapEndianess(mFormat);

    if (swap)
    {
        for (uint32_t i = 0; i < frames * mChannels; i++)
        {
            value = buf[i] - mLevelZero;
            swapEndianess(value);
            total += abs(value);
        }
    }
    else
    {
        for (uint32_t i = 0; i < frames * mChannels; i++)
        {
            value = buf[i] - mLevelZero;
            total += abs(value);
        }
    }

    mean  = total / (frames * mChannels);

    return mean;
}

template<typename E>
void SilenceDetector::swapEndianess(E& value)
{
    char *p = (char *)&value;
    unsigned int i, j;
    for(i = 0, j = (sizeof(E) - 1); j > i; i++, j--)
    {
        char tmp = p[i];
        p[i] = p[j];
        p[j] = tmp;
    }
}

void SilenceDetector::startProcessing()
{
    changeState(SdState::PROCESSING);
    mFirstActiveSampleDone = 1;
    mMean.clear();
}

void SilenceDetector::stopProcessing()
{
    changeState(SdState::STOPPED);
    mFirstActiveSampleDone = 0;
    mMean.clear();
}

void SilenceDetector::changeState(const SdState newState)
{
    mState = newState;
}

SdRet SilenceDetector::updateSdPrivateData()
{
    SdRet ret       = SdRet::OK;
    /* Updating Silence detector Private data */
    mMinFramesReq          = convertMsToFrames(mRate, MIN_PCM_DATA_REQ);
    mNoOfMeanValues        = (MIN_FADETIME_REQ / MIN_PCM_DATA_REQ );
    mFirstActiveSampleDone = 0;
    mMean.clear();
    parseEnvVariable();
    return ret;
}

void SilenceDetector::parseEnvVariable()
{
#define SD_MAX_PERCENT 15
    /* Reading environmental variable to set min mean percentage
     * SILENCE_DETECTOR_MIN_MEAN_PERCENTAGE=<Percentage>
     * Eg To set min mean as 2  percentage:
     *   "SILENCE_DETECTOR_MIN_MEAN_PERCENTAGE=2"
     * Note: The value Ranges between 0 to 15
     */
    char *sdMinMean = getenv("SILENCE_DETECTOR_MIN_MEAN_PERCENTAGE");
    uint64_t percent = DEFAULT_MIN_MEAN_PERCENTAGE;
    uint64_t maxValue = static_cast<uint64_t>(1 << ((getBytesPerSample(mFormat) * 8) - 1));

    if (sdMinMean != nullptr)
    {
        if (percent > SD_MAX_PERCENT)
        {
            percent = SD_MAX_PERCENT;
        }
    }
    mMinMeanAmplitude = (percent * maxValue) / 100;
}

void SilenceDetector::defaults()
{
    mFrameSize             = 0;
    mMinFramesReq          = 0;
    mNoOfMeanValues        = 0;
    mSilenceBuf            = nullptr;
    mFirstActiveSampleDone = 0;
    mFormat                = AudioFormat::UNKNOWN;
    mState                 = SdState::NONE;
    mChannels              = 0;
    mRate                  = 0;
    mFrames                = 0;
    mBytesPerSample        = 0;
    mLevelZero             = 0;
    mMinMeanAmplitude      = 0;
    mMean.clear();
}

SdRet SilenceDetector::isFormatSupported(const AudioFormat format)
{
    SdRet ret = SdRet::OK;
    switch (format)
    {
        case S8:
            mLevelZero = 0;
            break;
        case U8:
            mLevelZero = U8_LEVEL_ZERO;
            setSilenceData(static_cast<void*>(mSilenceBuf), static_cast<uint8_t>(mLevelZero), mFrames);
            break;
        case S16_LE:
            mLevelZero = 0;
            setSilenceData(static_cast<void*>(mSilenceBuf), static_cast<int16_t>(mLevelZero), mFrames);
            break;
        case S16_BE:
            mLevelZero = 0;
            setSilenceData(static_cast<void*>(mSilenceBuf), static_cast<int16_t>(mLevelZero), mFrames);
            break;
        case S32_LE:
            mLevelZero = 0;
            setSilenceData(static_cast<void*>(mSilenceBuf), static_cast<int32_t>(mLevelZero), mFrames);
            break;
        case S32_BE:
            mLevelZero = 0;
            setSilenceData(static_cast<void*>(mSilenceBuf), static_cast<int32_t>(mLevelZero), mFrames);
            break;
        case U16_LE:
            mLevelZero = U16_LEVEL_ZERO;
            setSilenceData(static_cast<void*>(mSilenceBuf), static_cast<uint16_t>(mLevelZero), mFrames);
            break;
        case U16_BE:
            mLevelZero = U16_LEVEL_ZERO;
            setSilenceData(static_cast<void*>(mSilenceBuf), static_cast<uint16_t>(mLevelZero), mFrames);
            break;
        case U32_LE:
            mLevelZero = U32_LEVEL_ZERO;
            setSilenceData(static_cast<void*>(mSilenceBuf), static_cast<uint32_t>(mLevelZero), mFrames);
            break;
        case U32_BE:
            mLevelZero = U32_LEVEL_ZERO;
            setSilenceData(static_cast<void*>(mSilenceBuf), static_cast<uint32_t>(mLevelZero), mFrames);
            break;
        default :
            ret = SdRet::UNSUPPORTED;
            break;
    }
    return ret;
}

bool SilenceDetector::isSwapEndianess(const AudioFormat format)
{
    bool isLe = SD_FALSE;
    bool ret  = SD_FALSE;

#ifdef SD_LITTLE_ENDIAN
    isLe = SD_TRUE;
#else
    isLe = SD_FALSE;
#endif
    switch (format)
    {
        case S8:
        case U8:
            break;
        case S16_LE:
        case U16_LE:
        case S32_LE:
        case U32_LE:
            ret = !isLe;
            break;
        case S16_BE:
        case S32_BE:
        case U16_BE:
        case U32_BE:
            ret = isLe;
            break;
        default :
            break;
    }
    return ret;
}

void SilenceDetector::clearSdPrivateData()
{
    freeInternalBuffers();
    defaults();
}

SdRet SilenceDetector::allocSilenceBuffer()
{
    mSilenceBuf = new uint8_t [mFrameSize];
    if (mSilenceBuf == nullptr)
    {
        return SdRet::NOMEM;
    }

    return SdRet::OK;
}

template<typename L>
void SilenceDetector::setSilenceData(void* buffer, const L level, const uint32_t frames)
{
    L* buf = static_cast<L*>(buffer);

    for (uint32_t i = 0; i < frames * mChannels ; i++)
    {
       *buf = level;
        buf++;
    }
}

void SilenceDetector::freeInternalBuffers()
{
    if (mSilenceBuf)
    {
        delete [] mSilenceBuf;
    }

    mSilenceBuf = nullptr;
}

/* Function definition for API's Inherited from Streaming class */

void SilenceDetector::error(const std::string& data) const
{
    mStreamingHandle.error(data);
}

void SilenceDetector::warning(const std::string& data) const
{
    mStreamingHandle.warning(data);
}

void SilenceDetector::info(const std::string& data) const
{
    mStreamingHandle.info(data);
}

void SilenceDetector::debug(const std::string& data) const
{
    mStreamingHandle.debug(data);
}

adit::utility::eLogLevel SilenceDetector::checkLogLevel() const
{
    return mStreamingHandle.checkLogLevel();
}

AudioState SilenceDetector::processing(unsigned char *in, unsigned char **out, uint32_t &frames)
{
    AudioState ret = mStreamingHandle.processing(in, out, frames);
    if (out != nullptr)
    {
        (void)process(*out, frames);
    }
    return ret;
}

void SilenceDetector::statistics(const StreamStatistics& status)
{
    mStreamingHandle.statistics(status);
}

void SilenceDetector::eostreaming(const AudioError error)
{
    mStreamingHandle.eostreaming(error);
}
